<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package StudioPress\Genesis
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

namespace StudioPress\Genesis\Upgrade;

/**
 * Upgrade class. Called when `db_version` Genesis setting is below 3001.
 *
 * @since 3.1.0
 */
class Upgrade_DB_3001 implements Upgrade_DB_Interface {
	/**
	 * Upgrade method.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 */
	public function upgrade() {
		$this->migrate_blog_pages();
		$this->migrate_archive_pages();
	}

	/**
	 * Helper method to access (& include/instantiate if necessary) $wp_filesystem global.
	 *
	 * @since 3.1.0
	 *
	 * @return object $wp_filesystem global.
	 */
	public function filesystem() {
		global $wp_filesystem;

		if ( is_null( $wp_filesystem ) ) {
			require_once ABSPATH . '/wp-admin/includes/file.php';
			WP_Filesystem();
		}

		return $wp_filesystem;
	}

	/**
	 * Migrate query_args and/or template for pages using page_blog.php template.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 */
	public function migrate_blog_pages() {
		$page_blog_ids = get_posts(
			[
				'post_type'  => 'page',
				'meta_key'   => '_wp_page_template',
				'meta_value' => 'page_blog.php',
				'nopaging'   => true,
				'fields'     => 'ids',
			]
		);

		if ( empty( $page_blog_ids ) ) {
			return;
		}

		if ( ! $this->blog_template_exists() ) {
			$this->create_blog_template();
		}

		$parameters = array_filter(
			[
				'cat'              => genesis_get_option( 'blog_cat', GENESIS_SETTINGS_FIELD, false ),
				'category__not_in' => genesis_get_option( 'blog_cat_exclude', GENESIS_SETTINGS_FIELD, false ),
				'showposts'        => genesis_get_option( 'blog_cat_num', GENESIS_SETTINGS_FIELD, false ),
			]
		);

		foreach ( $page_blog_ids as $page_blog_id ) {
			$existing_query_args = genesis_get_custom_field( 'query_args', $page_blog_id );
			$query_args          = wp_parse_args( $existing_query_args, $parameters );

			if ( $query_args ) {
				update_post_meta( $page_blog_id, 'query_args', urldecode( http_build_query( $query_args ) ) );
			}
		}
	}

	/**
	 * Generate page_archive.php template file for blogs using default Genesis page_archive.php.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 */
	public function migrate_archive_pages() {
		$page_archive_ids = get_posts(
			[
				'post_type'  => 'page',
				'meta_key'   => '_wp_page_template',
				'meta_value' => 'page_archive.php',
				'nopaging'   => true,
				'fields'     => 'ids',
			]
		);

		if ( empty( $page_archive_ids ) ) {
			return;
		}

		if ( ! $this->archive_template_exists() ) {
			$this->create_archive_template();
		}
	}

	/**
	 * Determine if the 'Blog' page template exists.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 *
	 * @return bool True if the 'Blog' template theme exists. False if else.
	 */
	public function blog_template_exists() {
		$templates = get_page_templates();

		return isset( $templates['Blog'] );
	}

	/**
	 * Determine if the 'Archive' page template is available.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 *
	 * @return bool True if the 'Archive' template theme exists. False if else.
	 */
	public function archive_template_exists() {
		$templates = get_page_templates();

		return isset( $templates['Archive'] );
	}

	/**
	 * Create the 'page_blog.php' file within child theme if missing.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 *
	 * @return int|bool|void Void if file exists, number of bytes written, or false if error.
	 */
	public function create_blog_template() {
		$directory_path = get_stylesheet_directory();
		$page_blog_path = "{$directory_path}/page_blog.php";

		if ( file_exists( $page_blog_path ) ) {
			return;
		}

		$content = <<<'CONTENT'
<?php
/**
 * Template Name: Blog
 */

genesis();
CONTENT;

		return $this->filesystem()->put_contents( $page_blog_path, $content );
	}

	/**
	 * Create the 'page_archive.php' file within child theme if missing.
	 *
	 * @since 3.0.0
	 * @since 3.1.0 Moved to class method.
	 *
	 * @return int|bool|void Void if file exists, number of bytes written, or false if error.
	 */
	public function create_archive_template() {
		$directory_path    = get_stylesheet_directory();
		$page_archive_path = "{$directory_path}/page_archive.php";

		if ( file_exists( $page_archive_path ) ) {
			return;
		}

		$content = <<<'CONTENT'
<?php
/**
 * Template Name: Archive
 */

remove_action( 'genesis_entry_content', 'genesis_do_post_content' );
add_action( 'genesis_entry_content', 'genesis_page_archive_content' );

function genesis_page_archive_content() {
    $heading = ( genesis_a11y( 'headings' ) ? 'h2' : 'h4' );

    genesis_sitemap( $heading );
}

genesis();
CONTENT;

		return $this->filesystem()->put_contents( $page_archive_path, $content );
	}
}
