<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Markup
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

/**
 * Output markup conditionally.
 *
 * Supported keys for `$args` are:
 *
 *  - `html5` (`sprintf()` pattern markup),
 *  - `context` (name of context),
 *  - `echo` (default is true).
 *
 * Applies a `genesis_markup_{context}` filter early to allow shortcutting the function.
 *
 * Applies a `genesis_markup_{context}_output` filter at the end.
 *
 * @since 1.9.0
 * @since 3.0.0 Removed xhtml argument key as xhtml support was removed.
 *
 * @param array $args {
 *     Contains markup arguments.
 *     @type string html5   Legacy HTML5 markup.
 *     @type string context Markup context.
 *     @type string open    Opening HTML markup.
 *     @type string close   Closing HTML markup.
 *     @type array  atts    Initial attributes to apply to `open`, before filters.
 *     @type string content Content to be placed between open and close HTML markup.
 *     @type bool   echo    Flag indicating whether to echo or return the resultant string.
 *     @type array  params  Additional information/data to pass to the various filters.
 * }
 * @return string|null Markup.
 */
function genesis_markup( $args = [] ) {

	$defaults = [
		'html5'   => '',
		'context' => '',
		'open'    => '',
		'close'   => '',
		'atts'    => [],
		'content' => '',
		'echo'    => true,
		'params'  => [],
	];

	$args = wp_parse_args( $args, $defaults );

	/**
	 * Filter to short circuit the markup API.
	 *
	 * @since 1.9.0
	 *
	 * @param bool  false Flag indicating short circuit content.
	 * @param array $args Array with markup arguments.
	 *
	 * @see genesis_markup $args Array.
	 */
	$pre = apply_filters( "genesis_markup_{$args['context']}", false, $args );
	if ( false !== $pre ) {

		if ( ! $args['echo'] ) {
			return $pre;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- We print html on purpose to render the page
		echo $pre;
		return null;

	}

	if ( $args['html5'] ) {

		$tag = $args['context'] ? sprintf( $args['html5'], genesis_attr( $args['context'] ) ) : $args['html5'];

		/**
		 * Legacy contextual filter to modify 'html5' output markup.
		 *
		 * @since 1.9.0
		 *
		 * @param string $tag  HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$tag = $args['context'] ? apply_filters( "genesis_markup_{$args['context']}_output", $tag, $args ) : $tag;

		if ( ! $args['echo'] ) {
			return $tag;
		}
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- We print html on purpose to render the page
		echo $tag;
		return null;

	}

	if ( $args['context'] ) {

		$open = $args['open'] ? sprintf( $args['open'], genesis_attr( $args['context'], $args['atts'], $args ) ) : '';

		/**
		 * Contextual filter to modify 'open' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $open HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$open = apply_filters( "genesis_markup_{$args['context']}_open", $open, $args );

		/**
		 * Contextual filter to modify 'close' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $close HTML tag being processed by the API.
		 * @param array  $args  Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$close = apply_filters( "genesis_markup_{$args['context']}_close", $args['close'], $args );

		/**
		 * Contextual filter to modify 'content'.
		 *
		 * @since 2.6.0
		 *
		 * @param string $content Content being passed through Markup API.
		 * @param array  $args  Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$content = apply_filters( "genesis_markup_{$args['context']}_content", $args['content'], $args );

	} else {

		$open    = $args['open'];
		$close   = $args['close'];
		$content = $args['content'];

	}

	if ( $open || $args['content'] ) {
		/**
		 * Non-contextual filter to modify 'open' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $open HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$open = apply_filters( 'genesis_markup_open', $open, $args );
	}

	if ( $close || $args['content'] ) {
		/**
		 * Non-contextual filter to modify 'close' markup.
		 *
		 * @since 2.4.0
		 *
		 * @param string $close HTML tag being processed by the API.
		 * @param array  $args Array with markup arguments.
		 *
		 * @see genesis_markup $args Array.
		 */
		$close = apply_filters( 'genesis_markup_close', $close, $args );
	}

	if ( $args['echo'] ) {
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- We print html on purpose to render the page
		echo $open . $content . $close;

		return null;
	}

	return $open . $content . $close;

}

/**
 * Merge array of attributes with defaults, and apply contextual filter on array.
 *
 * The contextual filter is of the form `genesis_attr_{context}`.
 *
 * @since 2.0.0
 *
 * @param string $context    The context, to build filter name.
 * @param array  $attributes Optional. Extra attributes to merge with defaults.
 * @param array  $args       Optional. Custom data to pass to filter.
 * @return array Merged and filtered attributes.
 */
function genesis_parse_attr( $context, $attributes = [], $args = [] ) {

	$defaults = [
		'class' => sanitize_html_class( $context ),
	];

	$attributes = wp_parse_args( $attributes, $defaults );

	// Contextual filter.
	return apply_filters( "genesis_attr_{$context}", $attributes, $context, $args );

}

/**
 * Build list of attributes into a string and apply contextual filter on string.
 *
 * The contextual filter is of the form `genesis_attr_{context}_output`.
 *
 * @since 2.0.0
 *
 * @param string $context    The context, to build filter name.
 * @param array  $attributes Optional. Extra attributes to merge with defaults.
 * @param array  $args       Optional. Custom data to pass to filter.
 * @return string String of HTML attributes and values.
 */
function genesis_attr( $context, $attributes = [], $args = [] ) {

	$attributes = genesis_parse_attr( $context, $attributes, $args );

	$output = '';

	// Cycle through attributes, build tag attribute string.
	foreach ( $attributes as $key => $value ) {

		if ( ! $value ) {
			continue;
		}

		if ( true === $value ) {
			$output .= esc_html( $key ) . ' ';
		} else {
			$output .= sprintf( '%s="%s" ', esc_html( $key ), esc_attr( $value ) );
		}
	}

	$output = apply_filters( "genesis_attr_{$context}_output", $output, $attributes, $context, $args );

	return trim( $output );

}

/**
 * Helper function for use as a filter for when you want to prevent a class from being automatically
 * generated and output on an element that is passed through the markup API.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes.
 * @return array Attributes with `class` set to empty string.
 */
function genesis_attributes_empty_class( $attributes ) {

	$attributes['class'] = '';

	return $attributes;

}

/**
 * Helper function for use as a filter for when you want to add screen-reader-text class to an element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes.
 * @return array Attributes with `screen-reader-text` added to classes
 */
function genesis_attributes_screen_reader_class( $attributes ) {

	$attributes['class'] .= ' screen-reader-text';

	return $attributes;

}

add_filter( 'genesis_attr_head', 'genesis_attributes_head' );
/**
 * Add attributes for head element.
 *
 * @since 2.2.0
 * @since 3.1.0 Moved microdata schema to `lib/functions/schema.php`.
 *
 * @param array $attributes Existing attributes for `head` element.
 * @return array Amended attributes for `head` element.
 */
function genesis_attributes_head( $attributes ) {

	$attributes['class'] = '';

	return $attributes;

}

add_filter( 'genesis_attr_body', 'genesis_attributes_body' );
/**
 * Add attributes for body element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to `lib/functions/schema.php`.
 *
 * @param array $attributes Existing attributes for `body` element.
 * @return array Amended attributes for `body` element.
 */
function genesis_attributes_body( $attributes ) {

	$attributes['class'] = implode( ' ', get_body_class() );

	return $attributes;

}

add_filter( 'genesis_attr_header-widget-area', 'genesis_attributes_header_widget_area' );
/**
 * Add attributes for header widget area element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for header widget area element.
 * @return array Amended attributes for header widget area element.
 */
function genesis_attributes_header_widget_area( $attributes ) {

	$attributes['class'] = 'widget-area header-widget-area';

	return $attributes;

}

add_filter( 'genesis_attr_breadcrumb-link-wrap-meta', 'genesis_attributes_breadcrumb_link_wrap_meta' );
/**
 * Add attributes for breadcrumb link wrap meta element.
 *
 * @since 2.7.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for breadcrumb link wrap meta element.
 * @return array Amended attributes for breadcrumb link wrap meta element.
 */
function genesis_attributes_breadcrumb_link_wrap_meta( $attributes ) {

	$attributes['class'] = '';

	return $attributes;

}

add_filter( 'genesis_attr_breadcrumb-link', 'genesis_attributes_breadcrumb_link', 10, 3 );
/**
 * Add attributes for breadcrumb link element.
 *
 * @since 2.7.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array  $attributes Existing attributes for breadcrumb link element.
 * @param string $context   Not used. Markup context (ie. `footer-widget-area`).
 * @param array  $args      Markup arguments.
 * @return array Amended attributes for breadcrumb link element.
 */
function genesis_attributes_breadcrumb_link( $attributes, $context, $args ) {

	$attributes['href'] = esc_url( $args['params']['href'] );

	return $attributes;

}

add_filter( 'genesis_attr_search-form', 'genesis_attributes_search_form' );
/**
 * Add attributes for search form.
 *
 * @since 2.2.0
 * @since 3.1.0 Moved microdata schema to `lib/functions/schema.php`.
 *
 * @param array $attributes Existing attributes for search form element.
 * @return array Amended attributes for search form element.
 */
function genesis_attributes_search_form( $attributes ) {

	$attributes['method'] = 'get';
	$attributes['action'] = home_url( '/' );
	$attributes['role']   = 'search';

	return $attributes;

}

add_filter( 'genesis_markup_search-form_content', 'genesis_markup_search_form_content' );
/**
 * Amend the search form content to include a meta tag (for schema).
 *
 * @since 2.7.0
 *
 * @param string $content Existing search form content.
 * @return string Potentially modified search form content.
 */
function genesis_markup_search_form_content( $content ) {

	$meta = [
		'open'    => '<meta %s>',
		'context' => 'search-form-meta',
		'echo'    => false,
	];

	return $content . genesis_markup( $meta );

}

add_filter( 'genesis_attr_search-form-meta', 'genesis_attributes_search_form_meta' );
/**
 * Add attributes for search form meta tag.
 *
 * @since 2.7.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for search form meta element.
 * @return array Amended attributes for search form meta element.
 */
function genesis_attributes_search_form_meta( $attributes ) {

	$attributes['class']   = '';
	$attributes['content'] = home_url( '/?s={s}' );

	return $attributes;

}

add_filter( 'genesis_markup_search-form-label_open', 'genesis_markup_search_form_label_control', 10, 2 );
add_filter( 'genesis_markup_search-form-label_close', 'genesis_markup_search_form_label_control', 10, 2 );
/**
 * Control the open/close tags for the search form label.
 *
 * Ensure that the label open/close tags get disabled if the label has no content.
 *
 * @since 2.7.0
 *
 * @param string $tag  Existing tag for search form label element.
 * @param array  $args Markup arguments.
 * @return string Potentially modified tag for search form label element.
 */
function genesis_markup_search_form_label_control( $tag, $args ) {

	if ( '' === $args['content'] ) {
		return '';
	}

	return $tag;

}

add_filter( 'genesis_attr_search-form-label', 'genesis_attributes_search_form_label', 10, 3 );
/**
 * Add attributes for search form label.
 *
 * @since 2.7.0
 *
 * @param array  $attributes Existing attributes for footer widget area wrapper elements.
 * @param string $context    Not used. Markup context (ie. `footer-widget-area`).
 * @param array  $args       Markup arguments.
 * @return array Amended attributes for search form label element.
 */
function genesis_attributes_search_form_label( $attributes, $context, $args ) {

	if ( isset( $args['params']['input_id'] ) ) {
		$attributes['for'] = $args['params']['input_id'];
	}

	$attributes['class'] .= ' screen-reader-text';

	return $attributes;

}

add_filter( 'genesis_attr_search-form-input', 'genesis_attributes_search_form_input', 10, 3 );
/**
 * Add attributes for search form input element.
 *
 * @since 2.7.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array  $attributes Existing attributes for footer widget area wrapper elements.
 * @param string $context    Not used. Markup context (ie. `footer-widget-area`).
 * @param array  $args       Markup arguments.
 * @return array Amended attributes.
 */
function genesis_attributes_search_form_input( $attributes, $context, $args ) {

	$attributes['type'] = 'search';
	$attributes['name'] = 's';

	foreach ( [ 'id', 'value', 'placeholder' ] as $param ) {
		if ( isset( $args['params'][ $param ] ) ) {
			$attributes[ $param ] = $args['params'][ $param ];
		}
	}

	return $attributes;

}

add_filter( 'genesis_attr_search-form-submit', 'genesis_attributes_search_form_submit', 10, 3 );
/**
 * Add attributes for search form submit element.
 *
 * @since 2.7.0
 *
 * @param array  $attributes Existing attributes for footer widget area wrapper elements.
 * @param string $context    Not used. Markup context (ie. `footer-widget-area`).
 * @param array  $args       Markup arguments.
 * @return array Amended attributes.
 */
function genesis_attributes_search_form_submit( $attributes, $context, $args ) {

	$attributes['type'] = 'submit';

	if ( isset( $args['params']['value'] ) ) {
		$attributes['value'] = $args['params']['value'];
	}

	return $attributes;

}

add_filter( 'genesis_attr_nav-primary', 'genesis_attributes_nav_primary' );
/**
 * Add attributes for primary navigation element.
 *
 * @since 2.6.0
 *
 * @param array $attributes Existing attributes for primary navigation element.
 * @return array Amended attributes for primary navigation element.
 */
function genesis_attributes_nav_primary( $attributes ) {

	$attributes['aria-label'] = __( 'Main', 'genesis' );

	return $attributes;

}

add_filter( 'genesis_attr_nav-secondary', 'genesis_attributes_nav_secondary' );
/**
 * Add attributes for secondary navigation element.
 *
 * @since 2.6.0
 *
 * @param array $attributes Existing attributes for secondary navigation element.
 * @return array Amended attributes for secondary navigation element.
 */
function genesis_attributes_nav_secondary( $attributes ) {

	$attributes['aria-label'] = __( 'Secondary', 'genesis' );

	return $attributes;

}

add_filter( 'genesis_attr_nav-link-wrap', 'genesis_attributes_nav_link_wrap' );
/**
 * Add attributes for the span wrap around navigation item links.
 *
 * @since 2.2.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for span wrap around navigation item links.
 * @return array Amended attributes for span wrap around navigation item links.
 */
function genesis_attributes_nav_link_wrap( $attributes ) {

	$attributes['class'] = '';

	return $attributes;

}

add_filter( 'genesis_attr_nav-link', 'genesis_attributes_nav_link' );
/**
 * Add attributes for the navigation item links.
 *
 * Since we're utilizing a filter that plugins might also want to filter, don't overwrite class here.
 *
 * @since 2.2.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @link https://github.com/studiopress/genesis/issues/1226
 *
 * @param array $attributes Existing attributes for navigation item links.
 * @return array Amended attributes for navigation item links.
 */
function genesis_attributes_nav_link( $attributes ) {

	$attributes['class'] = str_replace( 'nav-link', '', $attributes['class'] );

	return $attributes;

}

add_filter( 'genesis_attr_structural-wrap', 'genesis_attributes_structural_wrap' );
/**
 * Add attributes for structural wrap elements.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for structural wrap elements.
 * @return array Amended attributes for structural wrap elements.
 */
function genesis_attributes_structural_wrap( $attributes ) {

	$attributes['class'] = 'wrap';

	return $attributes;

}

add_filter( 'genesis_attr_content', 'genesis_attributes_content' );
/**
 * Add attributes for main content element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for `main` element.
 * @return array Attributes for `main` element.
 */
function genesis_attributes_content( $attributes ) {

	return $attributes;

}

add_filter( 'genesis_attr_taxonomy-archive-description', 'genesis_attributes_taxonomy_archive_description' );
/**
 * Add attributes for taxonomy archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for taxonomy archive description element.
 * @return array Amended attributes for taxonomy archive description element.
 */
function genesis_attributes_taxonomy_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description taxonomy-archive-description taxonomy-description';

	return $attributes;

}

add_filter( 'genesis_attr_author-archive-description', 'genesis_attributes_author_archive_description' );
/**
 * Add attributes for author archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for author archive description element.
 * @return array Amended attributes for author archive description element.
 */
function genesis_attributes_author_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description author-archive-description author-description';

	return $attributes;

}

add_filter( 'genesis_attr_cpt-archive-description', 'genesis_attributes_cpt_archive_description' );
/**
 * Add attributes for CPT archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for CPT archive description element.
 * @return array Amended attributes for CPT archive description element.
 */
function genesis_attributes_cpt_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description cpt-archive-description';

	return $attributes;

}

add_filter( 'genesis_attr_date-archive-description', 'genesis_attributes_date_archive_description' );
/**
 * Add attributes for date archive description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for date archive description element.
 * @return array Amended attributes for date archive description element.
 */
function genesis_attributes_date_archive_description( $attributes ) {

	$attributes['class'] = 'archive-description date-archive-description archive-date';

	return $attributes;

}

add_filter( 'genesis_attr_blog-template-description', 'genesis_attributes_blog_template_description' );
/**
 * Add attributes for blog template description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for blog template description element.
 * @return array Amended attributes for blog template description element.
 */
function genesis_attributes_blog_template_description( $attributes ) {

	$attributes['class'] = 'archive-description blog-template-description';

	return $attributes;

}

add_filter( 'genesis_attr_posts-page-description', 'genesis_attributes_posts_page_description' );
/**
 * Add attributes for posts page description element.
 *
 * @since 2.2.1
 *
 * @param array $attributes Existing attributes for posts page description element.
 * @return array Amended attributes for posts page description element.
 */
function genesis_attributes_posts_page_description( $attributes ) {

	$attributes['class'] = 'archive-description posts-page-description';

	return $attributes;

}

add_filter( 'genesis_attr_entry', 'genesis_attributes_entry' );
/**
 * Add attributes for entry element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for entry element.
 * @return array Amended attributes for entry element.
 */
function genesis_attributes_entry( $attributes ) {

	$attributes['class']      = implode( ' ', get_post_class() );
	$attributes['aria-label'] = the_title_attribute(
		[
			'echo' => false,
		]
	);

	return $attributes;

}

add_filter( 'genesis_attr_entry-image', 'genesis_attributes_entry_image' );
/**
 * Add attributes for entry image element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for entry image element.
 * @return array Amended attributes for entry image element.
 */
function genesis_attributes_entry_image( $attributes ) {

	$attributes['class'] = genesis_get_option( 'image_alignment' ) . ' post-image entry-image';

	return $attributes;

}

add_filter( 'genesis_attr_entry-image-link', 'genesis_attributes_entry_image_link' );
/**
 * Add attributes for entry image link element.
 *
 * @since 2.3.0
 *
 * @param array $attributes Existing attributes for entry image link element.
 * @return array Amended attributes for entry image link element.
 */
function genesis_attributes_entry_image_link( $attributes ) {

	$attributes['href']        = get_permalink();
	$attributes['aria-hidden'] = 'true';
	$attributes['tabindex']    = '-1';
	$attributes['class']       = 'entry-image-link';

	return $attributes;

}

add_filter( 'genesis_attr_singular-entry-image', 'genesis_attributes_singular_entry_image' );
/**
 * Add attributes for singular entry image element.
 *
 * @since 3.1.0
 *
 * @param array $attributes Existing attributes for singular entry image element.
 * @return array Amended attributes for singular entry image element.
 */
function genesis_attributes_singular_entry_image( $attributes ) {

	$attributes['class'] = 'singular-image entry-image';

	return $attributes;

}

add_filter( 'genesis_attr_entry-image-widget', 'genesis_attributes_entry_image_widget' );
/**
 * Add attributes for entry image element shown in a widget.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for entry image element shown in a widget.
 * @return array Amended attributes for entry image element shown in a widget.
 */
function genesis_attributes_entry_image_widget( $attributes ) {

	$attributes['class'] = 'entry-image attachment-' . get_post_type();

	return $attributes;

}

add_filter( 'genesis_attr_entry-author-link', 'genesis_attributes_entry_author_link' );
/**
 * Add attributes for entry author link element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for entry author link element.
 * @return array Amended attributes for entry author link element.
 */
function genesis_attributes_entry_author_link( $attributes ) {

	$attributes['rel'] = 'author';

	return $attributes;

}

add_filter( 'genesis_attr_entry-title-link', 'genesis_attributes_entry_title_link' );
/**
 * Add attributes for entry title link.
 *
 * @since 2.6.0
 *
 * @param array $attributes Existing attributes for entry title element.
 * @return array Amended attributes for entry title element.
 */
function genesis_attributes_entry_title_link( $attributes ) {

	$attributes['rel']  = 'bookmark';
	$attributes['href'] = get_permalink();

	return $attributes;

}

add_filter( 'genesis_attr_entry-meta-before-content', 'genesis_attributes_entry_meta' );
add_filter( 'genesis_attr_entry-meta-after-content', 'genesis_attributes_entry_meta' );
/**
 * Add attributes for entry meta elements.
 *
 * @since 2.1.0
 *
 * @param array $attributes Existing attributes for entry meta elements.
 * @return array Amended attributes for entry meta elements.
 */
function genesis_attributes_entry_meta( $attributes ) {

	$attributes['class'] = 'entry-meta';

	return $attributes;

}

add_filter( 'genesis_attr_archive-pagination', 'genesis_attributes_pagination' );
add_filter( 'genesis_attr_entry-pagination', 'genesis_attributes_pagination' );
add_filter( 'genesis_attr_adjacent-entry-pagination', 'genesis_attributes_pagination' );
add_filter( 'genesis_attr_comments-pagination', 'genesis_attributes_pagination' );
/**
 * Add attributes for pagination element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for pagination element.
 * @return array Amended attributes for pagination element.
 */
function genesis_attributes_pagination( $attributes ) {

	$attributes['class'] .= ' pagination';

	return $attributes;

}

add_filter( 'genesis_attr_entry-comments', 'genesis_attributes_entry_comments' );
/**
 * Add attributes for entry comments element.
 *
 * @since 2.0.0
 *
 * @param array $attributes Existing attributes for entry comments element.
 * @return array Amended attributes for entry comments element.
 */
function genesis_attributes_entry_comments( $attributes ) {

	$attributes['id'] = 'comments';

	return $attributes;

}

add_filter( 'genesis_attr_comment', 'genesis_attributes_comment' );
/**
 * Add attributes for single comment element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for single comment element.
 * @return array Amended attributes for single comment element.
 */
function genesis_attributes_comment( $attributes ) {

	$attributes['class'] = '';
	$attributes['id']    = 'article-comment-' . get_comment_ID();

	return $attributes;

}

add_filter( 'genesis_attr_comment-author-link', 'genesis_attributes_comment_author_link' );
/**
 * Add attributes for comment author link element.
 *
 * @since 2.1.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for comment author link element.
 * @return array Amended attributes for comment author link element.
 */
function genesis_attributes_comment_author_link( $attributes ) {

	$attributes['rel'] = 'external nofollow';

	return $attributes;

}

add_filter( 'genesis_attr_comment-time-link', 'genesis_attributes_comment_time_link', 10, 3 );
/**
 * Add attributes for comment time link element.
 *
 * @since 2.1.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for comment time link element.
 * @param array $context Not used. Markup context (ie. `footer-widget-area`).
 * @param array $args Arguments.
 * @return array Amended attributes for comment time link.
 */
function genesis_attributes_comment_time_link( $attributes, $context, $args ) {

	$comment            = $args['params']['comment'];
	$attributes['href'] = get_comment_link( $comment->comment_ID );

	return $attributes;

}

add_filter( 'genesis_attr_sidebar-primary', 'genesis_attributes_sidebar_primary' );
/**
 * Add attributes for primary sidebar element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for primary sidebar element.
 * @return array Amended attributes for primary sidebar element.
 */
function genesis_attributes_sidebar_primary( $attributes ) {

	$attributes['class']      = 'sidebar sidebar-primary widget-area';
	$attributes['role']       = 'complementary';
	$attributes['aria-label'] = __( 'Primary Sidebar', 'genesis' );

	return $attributes;

}

add_filter( 'genesis_attr_sidebar-secondary', 'genesis_attributes_sidebar_secondary' );
/**
 * Add attributes for secondary sidebar element.
 *
 * @since 2.0.0
 * @since 3.1.0 Moved microdata schema to lib/functions/schema.php.
 *
 * @param array $attributes Existing attributes for secondary sidebar element.
 * @return array Amended attributes for secondary sidebar element.
 */
function genesis_attributes_sidebar_secondary( $attributes ) {

	$attributes['class']      = 'sidebar sidebar-secondary widget-area';
	$attributes['role']       = 'complementary';
	$attributes['aria-label'] = __( 'Secondary Sidebar', 'genesis' );

	return $attributes;

}

add_filter( 'genesis_attr_footer-widget-area', 'genesis_attributes_footer_widget_area', 10, 3 );
/**
 * Add attributes for footer widget area wrapper elements.
 *
 * @since 2.5.0
 *
 * @param array  $attributes Existing attributes for footer widget area wrapper elements.
 * @param string $context    Not used. Markup context (ie. `footer-widget-area`).
 * @param array  $args       Markup arguments.
 * @return array Amended attributes for footer widget area wrapper elements.
 */
function genesis_attributes_footer_widget_area( $attributes, $context, $args ) {

	$column = ! empty( $args['params'] ) && ! empty( $args['params']['column'] ) ? $args['params']['column'] : 0;

	$attributes['class'] = sprintf( 'widget-area footer-widgets-%d ', $column ) . $attributes['class'];

	return $attributes;

}

/**
 * Add ID markup to the elements to jump to.
 *
 * @since 2.2.0
 *
 * @link https://gist.github.com/salcode/7164690
 */
function genesis_skiplinks_markup() {

	add_filter( 'genesis_attr_nav-primary', 'genesis_skiplinks_attr_nav_primary' );
	add_filter( 'genesis_attr_content', 'genesis_skiplinks_attr_content' );
	add_filter( 'genesis_attr_sidebar-primary', 'genesis_skiplinks_attr_sidebar_primary' );
	add_filter( 'genesis_attr_sidebar-secondary', 'genesis_skiplinks_attr_sidebar_secondary' );
	add_filter( 'genesis_attr_footer-widgets', 'genesis_skiplinks_attr_footer_widgets' );

}

/**
 * Add ID markup to primary navigation.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for primary navigation element.
 * @return array Amended attributes for primary navigation element.
 */
function genesis_skiplinks_attr_nav_primary( $attributes ) {

	$attributes['id'] = 'genesis-nav-primary';

	return $attributes;

}

/**
 * Add ID markup to main content area.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for `main` element.
 * @return array Amended attributes for `main` element.
 */
function genesis_skiplinks_attr_content( $attributes ) {

	$attributes['id'] = 'genesis-content';

	return $attributes;

}

/**
 * Add ID markup to primary sidebar.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for primary sidebar element.
 * @return array Amended attributes for primary sidebar element.
 */
function genesis_skiplinks_attr_sidebar_primary( $attributes ) {

	$attributes['id'] = 'genesis-sidebar-primary';

	return $attributes;

}

/**
 * Add ID markup to secondary sidebar.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for secondary sidebar element.
 * @return array Amended attributes for secondary sidebar element.
 */
function genesis_skiplinks_attr_sidebar_secondary( $attributes ) {

	$attributes['id'] = 'genesis-sidebar-secondary';

	return $attributes;

}

/**
 * Add ID markup to footer widget area.
 *
 * @since 2.2.0
 *
 * @param array $attributes Existing attributes for footer widget area element.
 * @return array Amended attributes for footer widget area element.
 */
function genesis_skiplinks_attr_footer_widgets( $attributes ) {

	$attributes['id'] = 'genesis-footer-widgets';

	return $attributes;

}

add_filter( 'genesis_attr_pagination-previous', 'genesis_adjacent_entry_attr_previous_post' );
/**
 * Add the alignleft class to the previous post link container.
 *
 * @since 2.7.0
 *
 * @param array $attributes Existing attributes for the previous post element.
 * @return array Amended attributes for the previous post element.
 */
function genesis_adjacent_entry_attr_previous_post( $attributes ) {
	$attributes['class'] .= ' alignleft';

	return $attributes;
}

add_filter( 'genesis_attr_pagination-next', 'genesis_adjacent_entry_attr_next_post' );
/**
 * Add the alignright class to the next post link container.
 *
 * @since 2.7.0
 *
 * @param array $attributes Existing attributes for the next post element.
 * @return array Amended attributes for the next post element.
 */
function genesis_adjacent_entry_attr_next_post( $attributes ) {
	$attributes['class'] .= ' alignright';

	return $attributes;
}
